/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'ComponentMouseDragger.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.tool;

import geofasc.swing.AbstractFigure;

import java.awt.Component;
import java.awt.Container;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JLayeredPane;
import javax.swing.SwingUtilities;

/**
 * The <code>ComponentMouseDragger</code> singleton class can be used to enable
 * mouse dragging on components. </p>
 * 
 * It is advisable to use this class only for components belonging to a
 * container having a null-layout since the components will be positioned
 * absolutely. If a dragged component's parent is a {@link JLayeredPane},
 * additionally that component is moved to the pane's front. Use
 * {@link #register(Component)} to enable and {@link #deregister(Component)} to
 * disable mouse dragging for a component.
 * 
 * @author Christian Wolf
 * @version 0.1 28/08/10
 * 
 */
public class ComponentMouseDragger {

	private static final ComponentMouseDragger sInstance = new ComponentMouseDragger();

	/**
	 * Deregisters the given component from this class and disables mouse
	 * dragging for it.
	 * 
	 * @param comp
	 *            the component to register
	 */
	public static void deregister(Component comp) {
		if (comp != null) {
			comp.removeMouseListener(sInstance.mComponentDragger);
			comp.removeMouseMotionListener(sInstance.mComponentDragger);
		}
	}

	/**
	 * Gets the singleton instance of this class.
	 * 
	 * @return the singleton object
	 */
	public static ComponentMouseDragger getInstance() {
		return sInstance;
	}

	/**
	 * Registers the given component to this class and enables mouse dragging
	 * for it.
	 * 
	 * @param comp
	 *            the component to register
	 */
	public static void register(Component comp) {
		if (comp != null) {
			deregister(comp);
			comp.addMouseListener(sInstance.mComponentDragger);
			comp.addMouseMotionListener(sInstance.mComponentDragger);
		}
	}

	private MouseAdapter mComponentDragger = new MouseAdapter() {

		Point mLastPressed = new Point();

		@Override
		public void mouseClicked(MouseEvent e) {
			if (e.getSource() instanceof Component) {
				Component clickedComponent = (Component) e.getSource();
				Container parent = clickedComponent.getParent();

				if (parent instanceof JLayeredPane)
					((JLayeredPane) parent).moveToFront(clickedComponent);
			}
		}

		@Override
		public void mouseDragged(MouseEvent e) {
			if (e.getSource() instanceof Component) {
				Component draggedComponent = (Component) e.getSource();
				Point p = SwingUtilities.convertPoint(draggedComponent, e
						.getPoint(), draggedComponent.getParent());

				if (draggedComponent.getParent() instanceof JLayeredPane)
					((JLayeredPane) draggedComponent.getParent())
							.moveToFront(draggedComponent);

				p.x -= mLastPressed.x;
				p.y -= mLastPressed.y;

				if (draggedComponent instanceof AbstractFigure)
					((AbstractFigure) draggedComponent).setFigureLocation(p);
				else
					draggedComponent.setLocation(p);
			}
		}

		@Override
		public void mousePressed(MouseEvent e) {
			mLastPressed.setLocation(e.getPoint());
		}
	}; // MouseAdapter

	/**
	 * Creates a new <code>ComponentMouseDragger</code>. This constructor is
	 * private to provide the singleton pattern.
	 */
	private ComponentMouseDragger() {

	}

}
