/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'RectangleModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;

import java.awt.Dimension;

/**
 * <code>RectangleModel</code> is the interface for the data model of a
 * rectangle figure. This interface is inherited from
 * {@link FigureModelExtended}. </p>
 * 
 * A rectangle is defined by its width and height (see
 * {@link FigureModelExtended#getSize()},
 * {@link FigureModelExtended#setSize(int, int)}). Besides the common visual
 * properties some extra ones can be set on it like drawing rounded corners.
 * 
 * @version 0.1 11/08/10
 * @author Christian Wolf
 * 
 */
public interface RectangleModel extends FigureModelExtended {

	/**
	 * Gets the arc height used when drawing this model's rectangle with rounded
	 * corners.
	 * 
	 * @return the current arc height
	 * @see #getArcWidth()
	 * @see #getArcSize()
	 */
	int getArcHeight();

	/**
	 * Gets the arc size (arc width and height) used when drawing this model's
	 * rectangle with rounded corners.
	 * 
	 * @return the current arc size
	 * @see #getArcWidth()
	 * @see #getArcHeight()
	 */
	Dimension getArcSize();

	/**
	 * Gets the arc width used when drawing this model's rectangle with rounded
	 * corners.
	 * 
	 * @return the current arc width
	 * @see #getArcHeight()
	 * @see #getArcSize()
	 */
	int getArcWidth();

	/**
	 * Determines whether to draw this model's rectangle with rounded corners.
	 * 
	 * @return true or false
	 * @see #setRoundedCorners(boolean)
	 */
	boolean hasRoundedCorners();

	/**
	 * Sets the arc height used when drawing this model's rectangle with rounded
	 * corners.
	 * 
	 * @param arcHeight
	 *            the new arc height
	 * @see #setArcWidth(int)
	 * @see #setArcSize(Dimension)
	 * @see #setArcSize(int, int)
	 */
	void setArcHeight(int arcHeight);

	/**
	 * Sets the arc size (arc width and height) used when drawing this model's
	 * rectangle with rounded corners.
	 * 
	 * @param arcSize
	 *            the new arc size
	 * @see #setArcSize(int, int)
	 * @see #setArcWidth(int)
	 * @see #setArcHeight(int)
	 */
	void setArcSize(Dimension arcSize);

	/**
	 * Sets the arc size (arc width and height) used when drawing this model's
	 * rectangle with rounded corners.
	 * 
	 * @param arcWidth
	 *            the new arc width
	 * @param arcHeight
	 *            the new arc height
	 * @see #setArcSize(Dimension)
	 * @see #setArcWidth(int)
	 * @see #setArcHeight(int)
	 */
	void setArcSize(int arcWidth, int arcHeight);

	/**
	 * Sets the arc width used when drawing this model's rectangle with rounded
	 * corners.
	 * 
	 * @param arcWidth
	 *            the arc width
	 * @see #setArcHeight(int)
	 * @see #setArcSize(Dimension)
	 * @see #setArcSize(int, int)
	 */
	void setArcWidth(int arcWidth);

	/**
	 * Sets whether this model's rectangle should have round corners.
	 * 
	 * @param roundCorners
	 *            true or false
	 * @see #hasRoundedCorners()
	 */
	void setRoundedCorners(boolean roundCorners);

}
