/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'DefaultPolylineModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * <code>DefaultPolylineModel</code> is the default implementation of
 * {@link PolylineModel}. This class is inherited from
 * {@link DefaultFigureModel}.
 * 
 * @version 0.2 19/03/11
 * @author Christian Wolf
 * 
 */
public class DefaultPolylineModel extends DefaultFigureModel implements
		PolylineModel {

	private List<Point> mPoints = Collections
			.synchronizedList(new ArrayList<Point>(3));

	private Dimension mSize = new Dimension();

	/** {@inheritDoc} */
	@Override
	public boolean addPoint(int x, int y) {
		return addPoint(new Point(x, y));
	}

	/** {@inheritDoc} */
	@Override
	public boolean addPoint(int index, int x, int y) {
		return addPoint(index, new Point(x, y));
	}

	/** {@inheritDoc} */
	@Override
	public synchronized boolean addPoint(int index, Point point) {
		boolean added = false;

		if (point != null) {
			try {
				mPoints.add(index, point);
				added = true;
			} catch (IndexOutOfBoundsException e) {
			}
		}

		if (added) {
			if (point.x > mSize.width)
				mSize.width = point.x;

			if (point.y > mSize.height)
				mSize.height = point.y;
			
			fireStateChanged();
		}

		return added;
	}

	/** {@inheritDoc} */
	@Override
	public synchronized boolean addPoint(Point point) {
		boolean added = false;

		if (point != null) {
			added = mPoints.add(point);
			if (added) {
				if (point.x > mSize.width)
					mSize.width = point.x;

				if (point.y > mSize.height)
					mSize.height = point.y;
				
				fireStateChanged();
			}
		}

		return added;
	}

	@Override
	public Rectangle getBounds() {
		return new Rectangle(getLocation(), mSize);
	}

	/** {@inheritDoc} */
	@Override
	public int getIndexOfPoint(int x, int y) {
		return mPoints.indexOf(new Point(x, y));
	}

	/** {@inheritDoc} */
	@Override
	public int getIndexOfPoint(Point point) {
		return mPoints.indexOf(point);
	}

	/** {@inheritDoc} */
	@Override
	public int getNumberOfPoints() {
		return mPoints.size();
	}

	/** {@inheritDoc} */
	@Override
	public Point getPoint(int index) {
		try {
			return mPoints.get(index);
		} catch (IndexOutOfBoundsException e) {
			return null;
		}
	}

	/** {@inheritDoc} */
	@Override
	public Point[] getPoints() {
		return mPoints.toArray(new Point[mPoints.size()]);
	}

	/** {@inheritDoc} */
	@Override
	public int[] getXPoints() {
		return toAWTPolygon().xpoints;
	}

	/** {@inheritDoc} */
	@Override
	public int[] getYPoints() {
		return toAWTPolygon().ypoints;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isClosed() {
		Point firstPoint = getPoint(0);
		Point lastPoint = getPoint(getNumberOfPoints() - 1);

		if (getNumberOfPoints() >= 3 && firstPoint != null && lastPoint != null)
			return firstPoint.equals(lastPoint);
		else
			return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean removePoint(int index) {
		boolean removed = false;

		try {
			if (mPoints.size() > 3) {
				mPoints.remove(index);
				removed = true;
			}
		} catch (IndexOutOfBoundsException e) {
		}

		if (removed) {
			updateSize();
			fireStateChanged();
		}

		return removed;
	}

	/** {@inheritDoc} */
	@Override
	public boolean removePoint(int x, int y) {
		return removePoint(new Point(x, y));
	}

	/** {@inheritDoc} */
	@Override
	public boolean removePoint(Point point) {
		boolean removed = false;

		if (mPoints.size() > 3 && (removed = mPoints.remove(point))) {
			if (removed) {
				updateSize();
				fireStateChanged();
			}
		}

		return removed;
	}

	/** {@inheritDoc} */
	@Override
	public Point setPoint(int index, int x, int y) {
		return setPoint(index, new Point(x, y));
	}

	/** {@inheritDoc} */
	@Override
	public Point setPoint(int index, Point point) {
		Point replaced = null;

		try {
			replaced = mPoints.set(index, point);
		} catch (IndexOutOfBoundsException e) {
		}

		if (replaced != null) {
			updateSize();
			fireStateChanged();
		}

		return replaced;
	}

	/** {@inheritDoc} */
	@Override
	public boolean setPoints(Point[] points) {
		synchronized (mPoints) {
			if (points != null && points.length >= 3) {
				mPoints.clear();
				for (Point p : points) {
					if (p != null)
						mPoints.add(p);
				}
				// TODO: Implement more efficient
				updateSize();
				fireStateChanged();
				return true;
			}
		}

		return false;
	}

	/** {@inheritDoc} */
	@Override
	public Polygon toAWTPolygon() {
		Polygon polygon = null;

		synchronized (mPoints) {
			int[] xPoints = new int[mPoints.size()];
			int[] yPoints = new int[mPoints.size()];

			int i = 0;
			Iterator<Point> it = mPoints.iterator();
			while (it.hasNext()) {
				Point p = it.next();
				xPoints[i] = p.x;
				yPoints[i] = p.y;
				i++;
			}

			polygon = new Polygon(xPoints, yPoints, xPoints.length);
		}

		return polygon;
	}

	private void updateSize() {
		synchronized (mPoints) {
			int newWidth = Integer.MIN_VALUE;
			int newHeight = Integer.MIN_VALUE;

			Iterator<Point> it = mPoints.iterator();
			while (it.hasNext()) {
				Point p = it.next();
				if (p.x > newWidth)
					newWidth = p.x;

				if (p.y > newHeight)
					newHeight = p.y;
			}
			mSize.setSize(newWidth, newHeight);
		}
	}
}
