/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'FigureDemo.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.demo;

import geofasc.swing.AbstractFigure;
import geofasc.swing.Circle;
import geofasc.swing.Ellipse;
import geofasc.swing.LineSegment;
import geofasc.swing.Polyline;
import geofasc.swing.Rectangle;
import geofasc.swing.Square;
import geofasc.swing.demo.TrafficLight.TrafficLightState;
import geofasc.swing.model.CircleModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.PolylineModel;
import geofasc.swing.plaf.BasicLineSegmentUI;
import geofasc.swing.plaf.BasicPolylineUI;
import geofasc.swing.tool.Canvas;
import geofasc.swing.tool.ComponentMouseDragger;
import geofasc.swing.tool.Frame;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.HeadlessException;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.SwingUtilities;

/**
 * <code>FiguresDemo</code> is a small demo presenting geofasc's components and
 * components composed by such ones.
 * 
 * @author Christian Wolf
 * 
 */
@SuppressWarnings("serial")
public class FiguresDemo extends Frame {

	public FiguresDemo(String title) throws HeadlessException {
		super(title);
	}

	public void demo() {
		// the frame's canvas
		Canvas canvas = getCanvas();

		// below the diverse figures are instantiated
		final Rectangle rectangle = new Rectangle(100, 50);
		rectangle.setRoundedCorners(true);
		rectangle.setArcSize(20, 40);
		rectangle.setFigureFilled(true);
		rectangle.setFigureFillColor(Color.BLUE);
		rectangle.setFigureLineColor(Color.RED);
		rectangle.setToolTipText("I'm a rectangle with rounded corners");
		ComponentMouseDragger.register(rectangle);
		canvas.add(rectangle);
		
		Thread t = new Thread() {
			
			@Override
			public void run() {
				rectangle.setFigureDirection(-25);
				int cnt = 0;
				int pixels = 3;
				
				while(true) {
					rectangle.moveFigureLocationBy(pixels);
					cnt++;
					
					if (cnt >= 60) {
						pixels = -pixels;
						cnt = 0;
						rectangle.turnFigureDirectionBy(-90);
					}
					
					try {
						Thread.sleep(50);
					}
					catch (Exception e) {
					}
				}
				
			}
		};
		t.start();
		
		Rectangle rectangle2 = new Rectangle(60, 30);
		rectangle2.setFigureFilled(true);
		rectangle2.setFigureFillColor(Color.YELLOW);
		rectangle2.setToolTipText("I'm another ordinary rectangle");
		ComponentMouseDragger.register(rectangle2);
		canvas.add(rectangle2);

		Square square = new Square(60);
		square.setToolTipText("I'm an ordinary square");
		ComponentMouseDragger.register(square);
		canvas.add(square);

		Ellipse ellipse = new Ellipse(50, 70);
		ellipse.setFigureLineColor(Color.CYAN);
		ellipse.setFigureFilled(true);
		ellipse.setFigureFillColor(new Color(50, 100, 89));
		ellipse.setToolTipText("I'm a filled ellipse");
		ComponentMouseDragger.register(ellipse);
		canvas.add(ellipse);

		Circle circle = new Circle();
		// here we access the circle through its model
		// just to demonstrate that the implementation
		// complies the MVC design pattern
		CircleModel model = circle.getModel();
		model.setRadius(60);
		model.setLineColor(Color.GREEN);
		circle.setToolTipText("I'm an ordinary circle");
		ComponentMouseDragger.register(circle);
		canvas.add(circle);

		final LineSegment line = new LineSegment(30, 20, 0, 200);
		// here we change the line's view just to
		// demonstrate that the implementation complies
		// the MVC design pattern
		
		final BasicLineSegmentUI ui1 = new BasicLineSegmentUI() {

			@Override
			protected void paintFigure(Graphics g, AbstractFigure f) {
				FigureModel figureModel = f.getModel();

				// draw the line's bounding box
				g.setColor(Color.BLUE);
				g.drawString("BB", 10, 20);
				g.drawRect(0, 0, figureModel.getWidth(), figureModel
						.getHeight());
				

				// draw line
				super.paintFigure(g, f);
			}

		};
		
		final BasicLineSegmentUI ui2 = new BasicLineSegmentUI() {

			@Override
			protected void paintFigure(Graphics g, AbstractFigure f) {
				// draw line
				super.paintFigure(g, f);
			}

		};
		
		JButton switchUI = new JButton("Switch line's UI");
		switchUI.setBounds(0, 0, 150, 30);
		canvas.add(switchUI);
		switchUI.addActionListener(new ActionListener() {
			
			@Override
			public void actionPerformed(ActionEvent e) {
				
				if (line.getUI() == ui1)
					line.setUI(ui2);
				else
					line.setUI(ui1);
				
			}
		});
		ComponentMouseDragger.register(switchUI);
		
		line
				.setToolTipText("I'm a line with a special view drawing my bounding box light gray");
		ComponentMouseDragger.register(line);
		canvas.add(line);

		// note the first and the last point have same coordinates
		// thus this polyline is a polygon
		Polyline polygon = new Polyline(new Point(30, 10), new Point(10, 50),
				new Point(50, 50), new Point(80, 150), new Point(40, 160),
				new Point(30, 10));
		polygon.setToolTipText("I'm an ordinary polygon");
		ComponentMouseDragger.register(polygon);
		canvas.add(polygon);

		Polyline polygon2 = new Polyline(new Point(30, 0), new Point(10, 50),
				new Point(50, 50), new Point(80, 150), new Point(40, 160),
				new Point(30, 0));
		// here we change the polygon2's view
		// just to demonstrate that the implementation complies the MVC design
		// pattern
		polygon2.setUI(new BasicPolylineUI() {

			@Override
			protected void paintFigure(Graphics g, AbstractFigure f) {
				// draw the polygon
				super.paintFigure(g, f);
				
				FigureModel figureModel = f.getModel();

				// draw the polygen's bounding box
				g.setColor(Color.LIGHT_GRAY);
				g.drawRect(0, 0, figureModel.getWidth(), figureModel
						.getHeight());

				// draw coordinates of corners at the same
				PolylineModel polyModel = ((Polyline) f).getModel();
				Point[] points = polyModel.getPoints();
				g.setColor(Color.RED);
				for (Point p : points) {
					g.fillOval(p.x - 3, p.y - 3, 6, 6);
				}

			}

		});
		polygon2
				.setToolTipText("I'm a filled polygon and with a special view drawing my corners highlighted");
		polygon2.setFigureFilled(true);
		polygon2.setFigureFillColor(Color.ORANGE);
		ComponentMouseDragger.register(polygon2);
		canvas.add(polygon2);

		Polyline polyline = new Polyline(new Point(30, 10), new Point(10, 50),
				new Point(60, 100), new Point(90, 10));
		polyline.setToolTipText("I'm an ordinary polyline");
		ComponentMouseDragger.register(polyline);
		canvas.add(polyline);

		// other components

		final TrafficLight trafficLight = new TrafficLight();
		trafficLight
				.setToolTipText("I'm a traffic light only consisting of geofasc components");
		ComponentMouseDragger.register(trafficLight);
		canvas.add(trafficLight);

		final TrafficLight trafficLight2 = new TrafficLight();
		trafficLight2.setState(TrafficLightState.GREEN);
		trafficLight2
				.setToolTipText("I'm another traffic light only consisting of geofasc components");
		ComponentMouseDragger.register(trafficLight2);
		canvas.add(trafficLight2);

		JButton button = new JButton("Next traffic light states");
		button.setSize(button.getPreferredSize());
		button
				.setToolTipText("I'm a Swing button tolerating geofasc components");
		button.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				trafficLight.setNextState();
				trafficLight2.setNextState();
			}
		});
		ComponentMouseDragger.register(button);
		canvas.add(button);

	}

	public static void main(String[] args) {
		SwingUtilities.invokeLater(new Runnable() {

			@Override
			public void run() {
				FiguresDemo frame = new FiguresDemo(
						"Demo of diverse figures and other components based on GeoFaSC");
				frame.setCenterLocation();
				frame.setVisible(true);
				frame.demo();
			}
		});

	}

}
