/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Frame.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.tool;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.Point;

import javax.swing.JFrame;

/**
 * A <code>Frame</code> (synonym is a window) is intended to be used as a
 * top-level container with a {@link Canvas} as its content pane.
 * 
 * @author Christian Wolf
 * @version 0.1 28/08/10
 * 
 */
@SuppressWarnings("serial")
public class Frame extends JFrame {

	/**
	 * Creates a new <code>Frame</code> with width 640 and height 480 in pixels.
	 * 
	 * @throws HeadlessException
	 */
	public Frame() throws HeadlessException {
		this(640, 480);
	}

	/**
	 * Creates a new <code>Frame</code> with the given size and with the
	 * top-left corner of the display as location.
	 * 
	 * @param width
	 *            the initial width of this frame
	 * @param height
	 *            the initial height of this frame
	 * @throws HeadlessException
	 */
	public Frame(int width, int height) throws HeadlessException {
		this(width, height, "Frame");
	}

	/**
	 * Creates a new <code>Frame</code> with the given size and at the given
	 * location.
	 * 
	 * @param x
	 *            the initial x-coordinate of this frame
	 * @param y
	 *            the initial x-coordinate of this frame
	 * @param width
	 *            the initial width of this frame
	 * @param height
	 *            the initial height of this frame
	 * @param title
	 *            the initial title of this frame
	 * @throws HeadlessException
	 */
	public Frame(int x, int y, int width, int height, String title)
			throws HeadlessException {
		super(title);
		setLocation(x, y);
		setSize(width, height);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setContentPane(new Canvas());
	}

	/**
	 * Creates a new <code>Frame</code> with the given size and with the
	 * top-left corner of the display as location.
	 * 
	 * @param width
	 *            the initial width of this frame
	 * @param height
	 *            the initial height of this frame
	 * @param title
	 *            the initial title of this frame
	 * @throws HeadlessException
	 */
	public Frame(int width, int height, String title) throws HeadlessException {
		this(0, 0, width, height, title);
	}

	/**
	 * Creates a new <code>Frame</code> with width 640 and height 480 in pixels
	 * and with the top-left corner of the display as location.
	 * 
	 * @param title
	 *            the initial title of this frame
	 * @throws HeadlessException
	 */
	public Frame(String title) throws HeadlessException {
		this(640, 480, title);
	}

	/**
	 * Convenience method for {@link Canvas#add(Component)}.
	 * 
	 * @param comp
	 *            the component to add
	 */
	public void addToCanvas(Component comp) {
		if (getCanvas() != null)
			getCanvas().add(comp);
	}

	/**
	 * Convenience method for {@link Canvas#add(Component[])}.
	 * 
	 * @param comps
	 *            the components to add
	 */
	public void addToCanvas(Component... comps) {
		if (getCanvas() != null)
			getCanvas().add(comps);
	}

	/**
	 * Sets this frame invisible and disposes it.
	 */
	public void close() {
		setVisible(false);
		dispose();
	}

	/**
	 * Gets the canvas of this frame used as the actual content pane.
	 */
	public Canvas getCanvas() {
		return (Canvas) getContentPane();
	}

	/** {@inheritDoc} */
	@Override
	public int getHeight() {
		return super.getHeight();
	}

	/** {@inheritDoc} */
	@Override
	public String getTitle() {
		return super.getTitle();
	}

	/** {@inheritDoc} */
	@Override
	public int getWidth() {
		return super.getWidth();
	}

	/**
	 * Sets the canvas of this frame used as the actual content pane.
	 * 
	 * @param canvas
	 *            the new canvas
	 */
	public void setCanvas(Canvas canvas) {
		setContentPane(canvas);
	}

	/**
	 * Sets the position of this frame to the center location of its display.
	 */
	public void setCenterLocation() {
		super.setLocationRelativeTo(null);
	}

	/**
	 * {@inheritDoc} </p>
	 * 
	 * Overridden to permit only panes of type {@link Canvas} as content panes.
	 * 
	 **/
	@Override
	public final void setContentPane(Container contentPane) {
		if (contentPane instanceof Canvas) {
			Canvas canvas = (Canvas) contentPane;
			
			Insets isets = getInsets();		
			int canvasWidth  = getWidth() - (isets.left + isets.right); 
			int canvasHeight = getHeight() - (isets.top + isets.bottom);
			
			canvas.setSize(canvasWidth, canvasHeight);
			super.setContentPane(canvas);
		}	
	}

	/**
	 * Sets the height of this frame.
	 * 
	 * @param height
	 *            the new height
	 */
	public void setHeight(int height) {
		setSize(getWidth(), height);
	}

	/** {@inheritDoc} */
	@Override
	public void setLocation(int x, int y) {
		super.setLocation(x, y);
	}

	/** {@inheritDoc} */
	@Override
	public void setLocation(Point p) {
		super.setLocation(p);
	}

	/** {@inheritDoc} */
	@Override
	public void setSize(Dimension d) {
		super.setSize(d);
	}

	/** {@inheritDoc} */
	@Override
	public void setSize(int width, int height) {
		super.setSize(width, height);
	}

	/** {@inheritDoc} */
	@Override
	public void setTitle(String title) {
		super.setTitle(title);
	}

	/** {@inheritDoc} */
	@Override
	public void setVisible(boolean b) {
		super.setVisible(b);
	}

	/**
	 * Sets the width of this frame.
	 * 
	 * @param width
	 *            new width
	 */
	public void setWidth(int width) {
		setSize(width, getHeight());
	}

	/** {@inheritDoc} */
	@Override
	public void toBack() {
		super.toBack();
	}

	/** {@inheritDoc} */
	@Override
	public void toFront() {
		super.toFront();
	}

}
