/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Pencil.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.demo;

import geofasc.swing.Polyline;
import geofasc.swing.tool.Canvas;

/**
 * A <code>Pencil</code> represents a tool that can draw something on a {@link Canvas}.
 * It can be put down and up, can be moved and turned around for drawing.
 * 
 * @author Christian Wolf
 * 
 */
public class Pencil {

	private double mAngle;
	private int mX;
	private int mY;
	private boolean mIsDown;
	private Polyline mPolyline;
	private Canvas mCanvas;
	
	public Pencil(Canvas canvas) {
		this(0, 0, canvas);
	}
	
	public Pencil(int x, int y, Canvas canvas) {
		mX = x;
		mY = y;
		mAngle = 0;
		mCanvas = canvas;
		mIsDown = false;
		putDown();
	}
	
	public void putUp() {
		mIsDown = false;
	}
	
	public void putDown() {
		if (mIsDown)
			return;
		
		mIsDown = true;
		mPolyline = new Polyline();
		mPolyline.addPoint(mX, mY);
		mCanvas.add(mPolyline);
	}
	
	public boolean isDown() {
		return mIsDown;
	}
	
	public boolean isUp() {
		return !mIsDown;
	}
	
	public void moveTo(int x, int y) {
		mX = x;
		mY = y;
		
		if (mIsDown)
			mPolyline.addPoint(mX, mY);
	}
	
	public void moveBy(int pixel) {
		int xNew = computeX(pixel);
		int yNew = computeY(pixel);
		
		moveTo(xNew, yNew);
	}

	public void turnAngleTo(double angle) {
		mAngle= angle;
  		while (mAngle < 0) 
  			mAngle += 360;
  		while (mAngle >= 720)
  			mAngle -= 360;
	}
	
	public void turnAngleBy(double angle) {
		mAngle = mAngle + angle;
  		while (mAngle < 0) 
  			mAngle += 360;
  		while (mAngle >= 720)
  			mAngle -= 360;
	}
	
	private int computeX(int pixel) {
  		double w = mAngle * Math.PI / 180;
  		double x = mX + pixel * Math.cos(w);
  		
  		return (int) Math.round(x);
	}
	
	private int computeY(int pixel) {
		double w = mAngle * Math.PI / 180;
  		double y = mY + pixel * Math.sin(w);
  		
  		return (int) Math.round(y);
	}
		
}


