/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Ellipse.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultEllipseModel;
import geofasc.swing.model.EllipseModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.FigureModelExtended;
import geofasc.swing.plaf.BasicEllipseUI;
import geofasc.swing.plaf.EllipseUI;
import geofasc.swing.plaf.RectangleUI;

import java.awt.Dimension;
import java.awt.Rectangle;

import javax.swing.UIManager;

/**
 * <code>Ellipse</code> is the component used for visualizing an ellipse figure.
 * See {@link EllipseModel} for more information on how an ellipse figure is
 * represented. This class is inherited from {@link AbstractFigure}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultEllipseModel
 * @see EllipseUI
 * @see BasicEllipseUI
 * 
 */
@SuppressWarnings("serial")
public class Ellipse extends AbstractFigure {

	private static final String uiClassID = "EllipseUI";

	/**
	 * Constructs a new empty <code>Ellipse</code> with width and height equal
	 * to 0.
	 */
	public Ellipse() {
		this(0, 0);
	}

	/**
	 * Constructs a new <code>Ellipse</code> with the given size.
	 * 
	 * @param width
	 *            the initial width of this ellipse
	 * @param height
	 *            the initial height of this ellipse
	 */
	public Ellipse(int width, int height) {
		this(0, 0, width, height);
	}

	/**
	 * Constructs a new <code>Ellipse</code> with the given bounds (size and
	 * location).
	 * 
	 * @param x
	 *            the initial x-location of this ellipse
	 * @param y
	 *            the initial y-location of this ellipse
	 * @param width
	 *            the initial width of this ellipse
	 * @param height
	 *            the initial height of this ellipse
	 */
	public Ellipse(int x, int y, int width, int height) {
		super(x, y);
		initModel(width, height);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultEllipseModel();
	}

	/** {@inheritDoc} */
	@Override
	public EllipseModel getModel() {
		return (EllipseModel) super.getModel();
	}

	/** {@inheritDoc} */
	@Override
	public EllipseUI getUI() {
		return (EllipseUI) ui;
	}

	/** {@inheritDoc} */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	private void initModel(int width, int height) {
		if (getModel() != null)
			getModel().setSize(width, height);
	}

	/**
	 * Sets the model of this ellipse.
	 * 
	 * @param model
	 *            the new model
	 */
	public void setModel(EllipseModel model) {
		super.setModel(model);
	}

	/**
	 * Sets the look and feel (L&F) object that renders this component.
	 * 
	 * @param ui
	 *            the new L&F object
	 */
	public void setUI(RectangleUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((RectangleUI) UIManager.getUI(this));
		} else {
			setUI(new BasicEllipseUI());
		}
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/**
	 * See {@link FigureModelExtended#moveHeightBy(int)}.
	 */
	public void moveFigureHeightBy(int dHeight) {
		if (getModel() != null)
			getModel().moveHeightBy(dHeight);
	}

	/**
	 * See {@link FigureModelExtended#moveSizeBy(Dimension)}.
	 */
	public void moveFigureSizeBy(Dimension dSize) {
		if (getModel() != null)
			getModel().moveSizeBy(dSize);
	}

	/**
	 * See {@link FigureModelExtended#moveSizeBy(int, int)}.
	 */
	public void moveFigureSizeBy(int dWidth, int dHeight) {
		if (getModel() != null)
			getModel().moveSizeBy(dWidth, dHeight);
	}

	/**
	 * See {@link FigureModelExtended#moveWidthBy(int)}.
	 */
	public void moveFigureWidthBy(int dWidth) {
		if (getModel() != null)
			getModel().moveWidthBy(dWidth);
	}

	/**
	 * See {@link FigureModelExtended#setBounds(int, int, int, int)}.
	 */
	public void setFigureBounds(int x, int y, int width, int height) {
		if (getModel() != null)
			getModel().setBounds(x, y, width, height);
	}

	/**
	 * See {@link FigureModelExtended#setBounds(Rectangle)}.
	 */
	public void setFigureBounds(Rectangle bounds) {
		if (getModel() != null)
			getModel().setBounds(bounds);
	}

	/**
	 * See {@link FigureModelExtended#setHeight(int)}.
	 */
	public void setFigureHeight(int height) {
		if (getModel() != null)
			getModel().setHeight(height);
	}

	/**
	 * See {@link FigureModelExtended#setSize(Dimension)}.
	 */
	public void setFigureSize(Dimension size) {
		if (getModel() != null)
			getModel().setSize(size);
	}

	/**
	 * See {@link FigureModelExtended#setSize(int, int)}.
	 */
	public void setFigureSize(int width, int height) {
		if (getModel() != null)
			getModel().setSize(width, height);
	}

	/**
	 * See {@link FigureModelExtended#setWidth(int)}.
	 */
	public void setFigureWidth(int width) {
		if (getModel() != null)
			getModel().setWidth(width);
	}

}
