/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Lamp.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.demo;

import geofasc.swing.Circle;

import java.awt.Color;

/**
 * A <code>Lamp</code> is a {@link Circle} with the common functionality of a
 * lamp. It can be turned on and off and the colors for these states can be set.
 * This class demonstrates how to build a new component based on a geofasc
 * component.
 * 
 * @author Christian Wolf
 * 
 */
@SuppressWarnings("serial")
public class Led extends Circle {

	private Color mOnColor;
	private Color mOffColor;
	private boolean mIsOn;

	public Led(int radius, Color onColor, Color offColor) {
		this(0, 0, radius, onColor, offColor, false);
	}

	public Led(int radius, Color onColor, Color offColor, boolean on) {
		this(0, 0, radius, onColor, offColor, on);
	}

	public Led(int x, int y, int radius, Color onColor, Color offColor) {
		this(x, y, radius, onColor, offColor, false);
	}

	public Led(int x, int y, int radius, Color onColor, Color offColor,
			boolean on) throws IllegalArgumentException {
		super(x, y, radius);

		if (onColor == null || offColor == null)
			throw new IllegalArgumentException(
					"onColor or offColor argument is null!");

		mOnColor = onColor;
		mOffColor = offColor;

		setOpaque(false);
		setFigureFilled(true);
		setOn(on);
	}

	public Color getOffColor() {
		return mOffColor;
	}

	public Color getOnColor() {
		return mOnColor;
	}

	public boolean isOff() {
		return !mIsOn;
	}

	public boolean isOn() {
		return mIsOn;
	}

	public void setOff() {
		setOn(false);
	}

	public void setOffColor(Color offColor) {
		if (offColor != null && offColor != mOffColor) {
			mOffColor = offColor;
			update();
		}
	}

	public void setOn() {
		setOn(true);
	}

	public void setOn(boolean on) {
		mIsOn = on;
		update();
	}

	public void setOnColor(Color onColor) {
		if (onColor != null && onColor != mOnColor) {
			mOnColor = onColor;
			update();
		}

	}

	public void toggleOnOff() {
		setOn(!isOn());
	}

	private void update() {
		if (mIsOn)
			setFigureFillColor(mOnColor);
		else
			setFigureFillColor(mOffColor);

	}

}
