/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Square.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultSquareModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.SquareModel;
import geofasc.swing.plaf.BasicRectangleUI;
import geofasc.swing.plaf.RectangleUI;


/**
 * <code>Square</code> is the component used for visualizing a square
 * figure. See {@link SquareModel} for more information on how a square
 * figure is represented. This class is inherited from {@link Rectangle}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultSquareModel
 * @see RectangleUI
 * @see BasicRectangleUI
 * 
 */
@SuppressWarnings("serial")
public class Square extends Rectangle {

	/**
	 * Constructs a new <code>Square</code> with surface area equal to 0.
	 */
	public Square() {
		this(0);
	}

	/**
	 * Constructs a new <code>Square</code> with the passed length.
	 * 
	 * @param length the initial length of the sides of this square
	 */
	public Square(int length) {
		this(0, 0, length);
	}

	/**
	 * Constructs a new <code>Square</code> with side of length
	 * <code>length</code>.
	 * 
	 * @param x the initial x-location of this square
	 * @param y the initial y-location of this square
	 * @param length initial the length of the sides of this square
	 */
	public Square(int x, int y, int length) {
		super(x, y, length, length);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultSquareModel();
	}

	/** {@inheritDoc} */
	@Override
	public SquareModel getModel() {
		return (SquareModel) super.getModel();
	}

	/**
	 * Sets the model of this square.
	 * 
	 * @param model
	 *            the new model
	 */
	public void setModel(SquareModel model) {
		super.setModel(model);
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/** See {@link SquareModel#getLength()} */
	public int getLength() {
		if (getModel() != null)
			return getModel().getLength();
		else
			return 0;
	}

	/** See {@link SquareModel#setLength(int)} */
	public void setLength(int length) {
		if (getModel() != null)
			getModel().setLength(length);
	}

}
